-- This script requires this View
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[ViewUserPRM]'))
EXEC dbo.sp_executesql @statement = N'
CREATE VIEW [dbo].[ViewUserPRM]
AS
SELECT		pe.PersonUid, u.ExternalID, 
			pe.FirstName, pe.LastName, pe.LastName + '', '' + pe.FirstName AS FullName, 
			pe.AkaFirstName, pe.AkaLastName, pe.MiddleName,
			u.Inactive, u.Login
FROM        dbo.Person pe INNER JOIN dbo.UserPRM u ON pe.PersonUid = u.PersonUid
' 
GO

--*********************************************************************
-- Homepage migration script. Analyzes prior version (18.6 or below)
-- tables & pulls config from RelUserPRMDesktopControl table.
-- A JSON blob of the transformed config is created per active user.
-- The blobs are stored in a table to be processed in two more stages
-- until the data is in its final homepage configuration destination.
--
-- This script is to be run as a POST migration step because the 19.0
-- tables need to be in place for the migration to work.
--Algorithm:
--STAGE1
--STORE ACTIVE USERS IN A TEMP TABLE
--FOR EACH ACTIVE USER
--BEGIN
--	FOR EACH CONFIG ROW OF THE CURRENT ACTIVE USER
--	BEGIN
--		EXTRACT DESKTOP CONFIG FOR PRIMARY & SECONDARY CONTROL
--		CONVERT CONTROL DATA TO JSON TILE EQUIVALENT
--		STORE JSON IN TEMP TABLE
--	END
--END (STAGE1)
--STAGE 2
--FOR EACH JSON CONFIG
--BEGIN
--	CONVERT JSON CONFIG TO INTERMEDIATE DATA & STORE IN CONFIG TEMP TABLE 		
--END
--END (STAGE2)
--STAGE 3
--FOR EACH ROW IN CONFIG TEMP TABLE
--BEGIN
--	EXTRACT DATA TO RESPECTIVE HOMEPAGE TABLES
--END
--END (STAGE3)
--*********************************************************************

--********************************************************************************
--***** NEED TO SET COMPATIBILITY TO 130 OR ABOVE TO SUPPORT JSONOPEN function
--ALTER DATABASE PRMHEAD SET COMPATIBILITY_LEVEL = 130;
--GO
--********************************************************************************

--******START STAGE 1******
DECLARE @xml xml
DECLARE @SettingsXml xml
DECLARE @CurrentUserUid uniqueidentifier
DECLARE @TileTypeUid uniqueidentifier
DECLARE @TileTypeName NVARCHAR(MAX) --tile type name
DECLARE @TileDataCfg NVARCHAR(MAX)
DECLARE @JSON_ApptQueueSortOrderingsBase NVARCHAR(MAX)
DECLARE @JSON_ApptQueueSortOrderingsDetailBase NVARCHAR(MAX)
DECLARE @IsAscending NVARCHAR(10)
DECLARE @AppointQueueSortOrderingId NVARCHAR(36)		= '0AD174FE-014E-496C-8BE1-3ECEDFEBE6D8'
DECLARE @AppointmentQueueTileTypeUid uniqueidentifier	= '72214A6B-49C5-4D11-8303-D08B23FF58F5'
DECLARE @DocumentLinkingTileTypeUid uniqueidentifier	= '973B8745-2931-46B9-8BEE-565385B64E84'
DECLARE @ExternalPatientsTileTypeUid uniqueidentifier	= '9AD5225E-4E56-4916-8E27-C7B1DC154520'
DECLARE @MessageCountsTileTypeUid uniqueidentifier		= '0D30D1A6-D56F-4404-BA37-221531FDFE9B'
DECLARE @MessageTileTypeUid uniqueidentifier			= '4EED7F56-6A65-41E5-B568-80A4701C33E1'
DECLARE @ActivePatientsTileTypeUid uniqueidentifier		= '75727057-82A0-412A-BD19-7EDCE6DC1DAF'
DECLARE @PatientAllergiesTileTypeUid uniqueidentifier	= '2D5F5A45-F25D-4B2F-A779-A8C0A800F6A4'
DECLARE @PatientInfoTileTypeUid uniqueidentifier		= 'D9687CFE-DCC9-435C-B379-5D24941DCCB5'
DECLARE @PatientMedicationsTileTypeUid uniqueidentifier	= '761F9AC2-0B0F-447C-9968-E79737F41874'
DECLARE @PatientVitalsTileTypeUid uniqueidentifier		= 'C9ECFED3-2CB1-44C1-80CA-4895322453E6'
DECLARE @PromotingInteroperablityTileTypeUid uniqueidentifier	= '8200F46B-A734-409B-9B14-25E8204EED48'
DECLARE @ProviderEMTileTypeUid uniqueidentifier			= '1947C8B8-050E-465C-8217-91EF525BC1EC'
DECLARE @RecentVisitsTileTypeUid uniqueidentifier		= '7F95EE5A-DC73-4F00-944B-27B8BF022894'
DECLARE @ReplicationMonitorTileTypeUid uniqueidentifier	= 'DFEE9D62-92D6-4068-845F-CB11C528D95D'
DECLARE @UnsentPrescriptonsTileTypeUid uniqueidentifier	= 'DAD2A405-8AF3-4831-8BC6-52B5C8959558'
DECLARE @VisitNotesTileTypeUid uniqueidentifier			= 'C7B0C52D-C2B3-43E8-A28F-8031D81F29BA'
DECLARE @WaitingListTileTypeUid uniqueidentifier		= 'C68E9C15-86C4-473F-A123-021F6CAE4B97'

DECLARE @ActiveUsers TABLE 
(
	PersonUid uniqueidentifier,
	FullName nvarchar(max),
	FirstName NVARCHAR(50),
	LastName NVARCHAR(50),
	LoginName nvarchar(50)
)

DECLARE @UserTracker TABLE 
(	
	LoginName nvarchar(50),
	Occurrence smallint
)

DECLARE @UserConfigsToProcessTbl TABLE 
(
	RelUserPRMDesktopControlUid uniqueidentifier,
	PrimaryDesktopControlUid uniqueidentifier,
	SecondaryDesktopControlUid uniqueidentifier,
	PrimaryControlName NVARCHAR(100),
	SecondaryControlName NVARCHAR(100),
	PrimaryData nvarchar(max),
	SecondaryData nvarchar(max),
	ColumnNumber INT,
	DisplayOrder INT
)

DECLARE @Stage1_JSONBlob TABLE 
(
	JSONBlobUid uniqueidentifier,
	JSONUserPRMUid uniqueidentifier,
	JSONUserName nvarchar(max),
	JSONBlob nvarchar(max)
)
DECLARE @ControlsData nvarchar(max)
DECLARE @CurrentDataRowUid uniqueidentifier
DECLARE @DisplayOrder INT
DECLARE @JSONCfg nvarchar(max)
DECLARE @UserConfigName NVARCHAR(102)
DECLARE @FirstName NVARCHAR(50)
DECLARE @LastName NVARCHAR(50)
DECLARE @LoginName nvarchar(50)
DECLARE @DesktopControlUid uniqueidentifier
DECLARE @ControlName NVARCHAR(100)

---store active users into temp table
INSERT INTO @ActiveUsers SELECT PersonUid, FullName, FirstName = FirstName, LastName = LastName, LoginName = Login FROM ViewUserPRM
WHERE Inactive = 0

DECLARE @JSON_CfgBase NVARCHAR(MAX)		= '{ "UserHPConfiguration": {"UserPRMUid": "", "HomepageName": "", "Tabs": []} }'
DECLARE @JSON_TabBase NVARCHAR(MAX)		= '{"comment": "", "Name": "", "Tiles": []}'
DECLARE @JSON_TileBase	NVARCHAR(MAX)	= '{"DesktopControlName": "", "TileName": "", "TileTypeUid": ""}'
SELECT @JSON_ApptQueueSortOrderingsBase = '{"SortType": "AppointmentQueueSort", "SortOrderingsDetails": []}' 
SELECT @JSON_ApptQueueSortOrderingsDetailBase = '{"Id": "", "Name": "", "ParameterValue": "", "IsAscending": "" }'
DECLARE @JSON_CfgWorking NVARCHAR(MAX)
DECLARE @JSON_TabDetail NVARCHAR(MAX)
DECLARE @JSON_TileDetail NVARCHAR(MAX)
DECLARE @TblRowCnt INT
DECLARE @SortOrderings NVARCHAR(MAX)
DECLARE @SortOrderingsDetail NVARCHAR(MAX)
DECLARE @SortType NVARCHAR(MAX)
DECLARE @JSON_Temp NVARCHAR(MAX)
DECLARE @TabIndex SMALLINT 
DECLARE @ContainsPrimaryControls BIT = 0
DECLARE @ContainsSecondaryControls BIT = 0
DECLARE @CurrentUserCfgRowUid uniqueidentifier
DECLARE @PrimaryDesktopControlUid uniqueidentifier
DECLARE @SecondaryDesktopControlUid uniqueidentifier

DECLARE @PrimaryData NVARCHAR(MAX)
DECLARE @SecondaryData NVARCHAR(MAX)
DECLARE @PrimaryDesktopControlName NVARCHAR(MAX)
DECLARE @SecondaryDesktopControlName NVARCHAR(MAX)
DECLARE @MaxProcessLoopCnt SMALLINT 
DECLARE @ProcessLoopCnt SMALLINT
DECLARE @ProcessingPrimaryCfg BIT
DECLARE @IncludeBottomPanel BIT
DECLARE @PanelIndex smallint --ColumnNumber in RelUserPRMDesktopControl
DECLARE @OccurrenceCnt SMALLINT
DECLARE @ConfigNameMaxLen SMALLINT = COL_LENGTH('dbo.ListGroupUIConfiguration', 'Name')
DECLARE @PaneCount SMALLINT

--process each active user's cfg & build a JSON blob to be stored in a temp table
WHILE EXISTS(SELECT * FROM @ActiveUsers)
BEGIN
	SELECT @ContainsPrimaryControls = 0	
	SELECT @ContainsSecondaryControls = 0
	SELECT @MaxProcessLoopCnt = 0
	SELECT @PaneCount = 1 --max of 3 as determined in usersettings

	SELECT TOP 1 @CurrentUserUid = PersonUid, @LoginName = LoginName, @FirstName=FirstName, @LastName=LastName FROM @ActiveUsers

	SELECT @SettingsXml = Settings from UserPRM WHERE PersonUid = @CurrentUserUid
	IF (@SettingsXml IS NOT NULL)
	BEGIN
		--PaneCount in user settings effectively equates to ColumnNumber in RelRelUserPRMDesktopControl which determines what panes will show
		SELECT @PaneCount = CONVERT(SMALLINT, @SettingsXml.value('(/Settings/Desktop/@PaneCount)[1]', 'nvarchar(max)'))	
	END
	
	--insert current user's cfg data to be processed into temp table	
	--NOTE: there is a case that a control has a NULL PrimaryData column like for Message control but 
	--when processing a Message control with no PrimaryData, it is handled accordingly (devops #113980)
	--#114547 skips all controls on a panel that is not to be shown
	INSERT INTO @UserConfigsToProcessTbl SELECT RelUserPRMDesktopControlUid, 
	PrimaryDesktopControlUid, SecondaryDesktopControlUid, 
	PrimaryControlName = pldc.Name, SecondaryControlName = sldc.Name, 
	PrimaryData, SecondaryData,	ColumnNumber, DisplayOrder FROM RelUserPRMDesktopControl rel
	LEFT JOIN ListDesktopControl pldc on pldc.DesktopControlUid = rel.PrimaryDesktopControlUid
	LEFT JOIN ListDesktopControl sldc on sldc.DesktopControlUid = rel.SecondaryDesktopControlUid
	WHERE UserPrmUid = @CurrentUserUid	
	AND ColumnNumber <= @PaneCount
	ORDER BY DisplayOrder asc

	--prevent users that do not have a cfg row in RelUserPRMDesktopControl
	if not exists(select * from @UserConfigsToProcessTbl)
	begin
		DELETE FROM @ActiveUsers WHERE PersonUid = @CurrentUserUid
		CONTINUE
	end
	
	--#114430 account for a user wout a login
	IF (@LoginName IS NULL)
		SELECT @LoginName = ''
	--#114430 create name for the config and account for duplicate login, firstname, lastname
	--if the current active user's Login, FirstInitial, LastName has duplicates through out the UserPrm table, need to track how many dups it finds
	--if dups are found (e.g. 2 dups for abailey Alex Bailey), the config name will be user1: "abailey A Bailey01" and user2: "abailey A Bailey02"
	SELECT @OccurrenceCnt = COUNT(*) FROM ViewUserPrm WHERE Inactive = 0 AND Login = @LoginName AND @LastName = LastName AND SUBSTRING(@FirstName, 1, 1) = SUBSTRING(FirstName, 1,1)
	IF (@OccurrenceCnt > 1)
	BEGIN
		IF (NOT EXISTS(SELECT * FROM @UserTracker WHERE LoginName = @LoginName))
		BEGIN
			--first time to add to UserTracker
			INSERT INTO @UserTracker (LoginName, Occurrence)
			VALUES(@LoginName, 1)
			SELECT @OccurrenceCnt = 1
		END
		ELSE
		BEGIN
			--Nth time to add to UserTracker
			SELECT @OccurrenceCnt = Occurrence FROM @UserTracker WHERE LoginName = @LoginName
			UPDATE @UserTracker SET Occurrence = @OccurrenceCnt + 1 WHERE LoginName = @LoginName
			SELECT @OccurrenceCnt = Occurrence FROM @UserTracker WHERE LoginName = @LoginName
		END
		SELECT @UserConfigName = @LoginName + ' ' + SUBSTRING(@FirstName, 1, 1) + ' ' + @LastName + '0' + CONVERT(VARCHAR(5), @OccurrenceCnt)
	END
	ELSE
		SELECT @UserConfigName = @LoginName + ' ' + SUBSTRING(@FirstName, 1, 1) + ' ' + @LastName
	--#114430 truncate configname to max len if needed
	SELECT @UserConfigName = Left(@UserConfigName, @ConfigNameMaxLen)
	
	--SELECT @UserConfigName AS CONFIGNAME

	--#114008 determine if user settings has bottom panel specified
	SELECT @IncludeBottomPanel = 0		
	--SELECT @SettingsXml = Settings from UserPRM WHERE PersonUid = @CurrentUserUid
	IF (@SettingsXml IS NOT NULL)
	BEGIN
		DECLARE @value nvarchar(10)
		SELECT @value = @SettingsXml.value('(/Settings/Desktop/@PaneBottom)[1]', 'nvarchar(max)')
		IF (LOWER(@value) = 'true')
			SELECT @IncludeBottomPanel = 1
	END

	--indicate if primary or secondary control data present as it flags which Tab[] to access
	IF (EXISTS(SELECT * FROM @UserConfigsToProcessTbl WHERE PrimaryData IS NOT NULL))
		SELECT @ContainsPrimaryControls = 1
	IF (EXISTS(SELECT * FROM @UserConfigsToProcessTbl WHERE SecondaryData IS NOT NULL))
		SELECT @ContainsSecondaryControls = 1

	--@JSON_CfgWorking is what holds the base template for the full JSON
	--as cfg is processed & is continuously being modified
	SELECT @JSON_CfgWorking = @JSON_CfgBase

	IF EXISTS(SELECT * FROM @UserConfigsToProcessTbl)
	BEGIN
		SELECT @JSON_CfgWorking = JSON_MODIFY(@JSON_CfgWorking, '$.UserHPConfiguration.UserPRMUid', CONVERT(VARCHAR(36), @CurrentUserUid))
		SELECT @JSON_CfgWorking = JSON_MODIFY(@JSON_CfgWorking, '$.UserHPConfiguration.HomepageName', @UserConfigName)
		
		--initialize the Tab[] based on presence of Primary & Secondary controls being present
		IF (@ContainsPrimaryControls = 1)
		BEGIN
			SELECT @JSON_TabDetail = @JSON_TabBase
			SELECT @JSON_TabDetail = JSON_MODIFY(@JSON_TabDetail, '$.comment', 'Primary Desktop Controls')
			SELECT @JSON_TabDetail = JSON_MODIFY(@JSON_TabDetail, '$.Name', 'Home')
			SELECT @JSON_CfgWorking = JSON_MODIFY(@JSON_CfgWorking, 'append $.UserHPConfiguration.Tabs', JSON_QUERY(@JSON_TabDetail))
		END
		IF (@ContainsSecondaryControls = 1)
		BEGIN
			SELECT @JSON_TabDetail = @JSON_TabBase
			SELECT @JSON_TabDetail = JSON_MODIFY(@JSON_TabDetail, '$.comment', 'Secondary Desktop Controls')
			SELECT @JSON_TabDetail = JSON_MODIFY(@JSON_TabDetail, '$.Name', 'Home2')
			SELECT @JSON_CfgWorking = JSON_MODIFY(@JSON_CfgWorking, 'append $.UserHPConfiguration.Tabs', JSON_QUERY(@JSON_TabDetail))
		END

		--for each row of a user's cfgs, create Tile details & apply to appropriate Tab[]
		WHILE EXISTS(SELECT * FROM @UserConfigsToProcessTbl)
		BEGIN
			SELECT TOP 1 @CurrentUserCfgRowUid = RelUserPRMDesktopControlUid, 
			@PrimaryDeskTopControlUid = PrimaryDesktopControlUid, 
			@SecondaryDesktopControlUid = SecondaryDesktopControlUid,
			@PrimaryData = PrimaryData,	
			@SecondaryData = SecondaryData,
			@PrimaryDesktopControlName = PrimaryControlName,
			@SecondaryDesktopControlName = SecondaryControlName,
			@PanelIndex = ColumnNumber
			FROM @UserConfigsToProcessTbl
			ORDER BY DisplayOrder ASC

			--#114371 skip a user who has no cfg whatsoever from being processed. user will see default cfg when they open homepage.
			--#114008 no bottom panel indicated for a control at ColumnNumber = 0 so skip this control
			IF ((@PrimaryData IS NULL AND @SecondaryData IS NULL) OR (@PanelIndex = '0' AND @IncludeBottomPanel = 0))
			BEGIN
				DELETE FROM @UserConfigsToProcessTbl WHERE RelUserPRMDesktopControlUid = @CurrentUserCfgRowUid
				CONTINUE
			END

			SELECT @MaxProcessLoopCnt = 2

			--process accordingly & create Tile data for the appropriate Tab[]
			SELECT @ProcessLoopCnt = 0
			SELECT @ProcessingPrimaryCfg = 1

			WHILE(@ProcessLoopCnt < @MaxProcessLoopCnt)
			BEGIN
				BEGIN
					IF (@ProcessingPrimaryCfg = 1) 
						SELECT @DesktopControlUid = @PrimaryDesktopControlUid
					ELSE
						SELECT @DesktopControlUid = @SecondaryDesktopControlUid
					
					--set TileTypeUid do determine what Tile to cfg from the current control config
					SET @TileTypeUid = 
					CASE 
						WHEN @DeskTopControlUid = '10101010-0000-0000-1000-000000000003' THEN @MessageTileTypeUid --Messages Control/Messages Tile
						WHEN @DeskTopControlUid = 'DB496E0C-9B07-4C28-BD72-4186ED58C623' THEN @MessageCountsTileTypeUid --Message Count Control/Message Count Tile
						WHEN @DesktopControlUid = '11576FEF-330C-47C7-A820-B7ADEC9F19A5' THEN @AppointmentQueueTileTypeUid --Appt Queue Control/Appt Queue Tile
						WHEN @DesktopControlUid = '10101010-0000-0000-1000-000000000001' THEN @AppointmentQueueTileTypeUid --Calendar Control/Appt Queue Tile
						WHEN @DesktopControlUid = 'A5156440-5671-44C6-B729-F24235F897B0' THEN @PromotingInteroperablityTileTypeUid --Meaningful Use Control/Promoting InteroperabilityAppt Tile
						WHEN @DesktopControlUid = 'AF06352F-98D4-4306-844D-F27E60AEB1C4' THEN @ExternalPatientsTileTypeUid --External Patient control/External Patients tile
						WHEN @DesktopControlUid = 'C7125D68-F370-4D5A-90DE-E78EAFB990C9' THEN @DocumentLinkingTileTypeUid --Document linking control/Document linking tile
						WHEN @DesktopControlUid = '395DB65B-3D0C-4A54-9295-A0E3C15B0F5D' THEN @VisitNotesTileTypeUid --Visit Notes control/ Visit Notes
						WHEN @DesktopControlUid = '10101010-0000-0000-1000-000000000002' THEN @WaitingListTileTypeUid --Waiting List control/Waiting List
						WHEN @DesktopControlUid = 'D88E6C6B-7701-4AD6-B150-E633220DC44F' THEN @ProviderEMTileTypeUid --Provider E&M control/Provider E&M tile
						WHEN @DesktopControlUid = 'E986820D-5313-4283-9065-0751A07B3D11' THEN @ReplicationMonitorTileTypeUid --Replication Monitor control/Replication Monitor tile
						WHEN @DesktopControlUid = '27319965-6A9B-4ECE-A1AA-BCCEE0E7ADA2' THEN @UnsentPrescriptonsTileTypeUid --Prescription Queue control/Unsent Prescriptions
						ELSE NULL
					END
				END

				--DEBUG
				--SELECT @TileTypeUid AS 'CURRENT TILETYPEUID', @DesktopControlUid AS 'CurrentDESKTOPCONTROLUID'

				--create TileDetail
				IF (@TileTypeUid IS NOT NULL)
				BEGIN
					SELECT @TileTypeName = Name FROM ListTileType WHERE TileTypeUid = @TileTypeUid

					--some controls have either just a <root/> or null for the column so level the playing field here
					SELECT @xml = '<root/>' 
					IF (@ProcessingPrimaryCfg = 1)
					BEGIN
						IF (@PrimaryData IS NOT NULL)
							SELECT @xml = @PrimaryData
						SELECT @ControlName = @PrimaryDesktopControlName
					END
					ELSE
					BEGIN
						IF (@SecondaryData IS NOT NULL)
							SELECT @xml = @SecondaryData
						SELECT @ControlName = @SecondaryDesktopControlName
					END

					--set the basic details for the current tile's details to be saved as JSON
					SELECT @JSON_TileDetail = @JSON_TileBase
					SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.DesktopControlName', @ControlName)
					SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.TileName', @TileTypeName)
					SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.TileTypeUid', Convert(Varchar(36), @TileTypeUid))
					
					--handle setting specific properties per tile as they differ from tile to tile
					IF (@TileTypeUid = @PromotingInteroperablityTileTypeUid)
					BEGIN
						--extract timespanUid value from xml and add to tile detail
						SELECT @JSON_Temp = @xml.value('(/root/MeaningfulUse/@timespanUid)[1]', 'nvarchar(max)')
						IF (LEN(@JSON_Temp) > 0)
							SELECT @JSON_Temp = '[' + '"' +  @xml.value('(/root/MeaningfulUse/@timespanUid)[1]', 'nvarchar(max)') + '"' + ']'
						ELSE
							SELECT @JSON_Temp = '[]'
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.timeSpanSelected', JSON_QUERY(@JSON_Temp));
						--extract providersUid
						SELECT @JSON_Temp = @xml.value('(/root/MeaningfulUse/@providersUid)[1]', 'nvarchar(max)')
						IF (LEN(@JSON_Temp) > 0)
							SELECT @JSON_Temp = '[' + Concat('"', Replace(@xml.value('(/root/MeaningfulUse/@providersUid)[1]', 'nvarchar(max)'), ';', '","'), '"') + ']'
						ELSE
							SELECT @JSON_Temp = '[]'
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.selectedProviderIds', JSON_QUERY(@JSON_Temp));
					END
					IF (@TileTypeUid = @ExternalPatientsTileTypeUid)
					BEGIN
						--#114269 - no filteruid is defined at time of migration
						--extract filteruid
						SELECT @JSON_Temp = @xml.value('(/root/Filter/@filteruid)[1]', 'nvarchar(max)')
						IF (LEN(@JSON_Temp) > 0)
							SELECT @JSON_Temp = '[' + '"' +  @JSON_Temp + '"' + ']'
						ELSE
							SELECT @JSON_Temp = '[]'
						--add selectedExternalPatientIds property into Tile
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.selectedExternalPatientIds', JSON_QUERY(@JSON_Temp));
					END
					IF (@TileTypeUid = @MessageTileTypeUid)
					BEGIN		
						--devops #113980 - a Message control that does not have a default filter specified in RelUserPrmDesktopControl, 
						--will have no xml whatsoever (i.e. PrimaryData is null) but @xml is given a <root/>.
						IF (@xml.value('(/root/Filter/@filteruid)[1]', 'nvarchar(max)') IS NOT NULL)
							SELECT @JSON_Temp = '[' + '"' +  @xml.value('(/root/Filter/@filteruid)[1]', 'nvarchar(max)') + '"' + ']'
						ELSE
							SELECT @JSON_Temp = '[]'

						--add filterids property into Tile
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.filterIds', JSON_QUERY(@JSON_Temp));
					END
					IF (@TileTypeUid = @MessageCountsTileTypeUid)
					BEGIN		
						--extract filteruid
						SELECT @JSON_Temp = Concat('"', Replace(@xml.value('(/root/Filter/@filteruid)[1]', 'nvarchar(max)'), ';', '","'), '"')
						SELECT @JSON_Temp = '[' + @JSON_Temp + ']'
						--add messageFilters property into Tile
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.messageFilters', JSON_QUERY(@JSON_Temp));
					END
					IF (@TileTypeUid = @AppointmentQueueTileTypeUid) -- destination tile to build is Appt Queue tile
					BEGIN
						IF (@DesktopControlUid = '11576FEF-330C-47C7-A820-B7ADEC9F19A5')
						BEGIN
							--Prior cfg is for an appt queue tile. It's config is in HomeHomeDashboardUserConfiguration table. TileInstance is really
							--a FK to RelUserPRMDesktopControl row and its IsPrimaryControl setting needs to match the current state of processing a primary or secondary control
							SELECT @TileDataCfg = TileData FROM HomeDashboardUserConfiguration WHERE (UserPrmUid = @CurrentUserUid AND TileInstanceUid = @CurrentUserCfgRowUid
								AND IsPrimaryControl = @ProcessingPrimaryCfg)
							IF (@TileDataCfg IS NOT NULL)
							BEGIN
								--appointmentStatusSelectedIds/appointmentStatusFilterIds
								SELECT @JSON_Temp = JSON_QUERY(@TileDataCfg, '$.appointmentQueueFilter.appointmentStatusSelectedIds')
								IF (@JSON_Temp IS NOT NULL)
									SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.appointmentStatusFilterIds', JSON_QUERY(@JSON_Temp));
								--calendarSelectedIds/calendarFilterIds
								SELECT @JSON_Temp = JSON_QUERY(@TileDataCfg, '$.appointmentQueueFilter.calendarSelectedIds')
								IF (@JSON_Temp IS NOT NULL)
									SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.calendarFilterIds', JSON_QUERY(@JSON_Temp));
								--roommSelectedIds/roomFilterIds
								SELECT @JSON_Temp = JSON_QUERY(@TileDataCfg, '$.appointmentQueueFilter.roomSelectedIds')
								IF (@JSON_Temp IS NOT NULL)
									SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.roomFilterIds', JSON_QUERY(@JSON_Temp));
								--appointmentTypeSelectedIds/appointmentTypeFilterIds
								SELECT @JSON_Temp = JSON_QUERY(@TileDataCfg, '$.appointmentQueueFilter.appointmentTypeSelectedIds')
								IF (@JSON_Temp IS NOT NULL)
									SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.appointmentTypeFilterIds', JSON_QUERY(@JSON_Temp));

								--get sortdirection from cfg
								SELECT @IsAscending = JSON_VALUE(@TileDataCfg, '$.appointmentQueueSortOptions.isSortAscending')
								SELECT @SortType = JSON_VALUE(@TileDataCfg, '$.appointmentQueueSortOptions.sortType')

								--appt queue sorting {"SortType": "AppointmentQueueSort", "SortOrderingsDetails": []}
								--SortOrderingsDetail {"Id": "", "Name": "", "ParameterValue": "", "IsAscending": "" }
								SELECT @SortOrderings = @JSON_ApptQueueSortOrderingsBase
								SELECT @SortOrderingsDetail = @JSON_ApptQueueSortOrderingsDetailBase

								--setup sortorderingsdetail
								SELECT @SortOrderingsDetail = JSON_MODIFY(@SortOrderingsDetail, '$.Id', @AppointQueueSortOrderingId)
								SELECT @SortOrderingsDetail = JSON_MODIFY(@SortOrderingsDetail, '$.Name', 'Appointment Time')
								SELECT @SortOrderingsDetail = JSON_MODIFY(@SortOrderingsDetail, '$.ParameterValue', 'StartDateTime')
								SELECT @SortOrderingsDetail = JSON_MODIFY(@SortOrderingsDetail, '$.IsAscending', @IsAscending)

								--assign sortOrderingsDetail to sortOrderings
								SELECT @SortOrderings = JSON_MODIFY(@SortOrderings, 'append $.SortOrderingsDetails', JSON_QUERY(@SortOrderingsDetail))
								
								--apply sortOrderings to tile
								SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.sortOrderings', JSON_QUERY(@SortOrderings));
							END
						END
						IF (@DesktopControlUid = '10101010-0000-0000-1000-000000000001')
						BEGIN
							--Prior cfg is a for Calendar control
							SELECT @JSON_Temp = @xml.value('(/root/Calendar/@calendaruids)[1]', 'nvarchar(max)')
							IF (LEN(@JSON_Temp) > 0)
							BEGIN
								SELECT @JSON_Temp = Concat('"', Replace(@JSON_Temp, ';', '","'), '"')
								SELECT @JSON_Temp = '[' + @JSON_Temp + ']'
							END
							ELSE
								SELECT @JSON_Temp = '[]'
							SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.calendarFilterIds', JSON_QUERY(@JSON_Temp));
						END
					END
					IF (@TileTypeUid = @DocumentLinkingTileTypeUid)
					BEGIN						
						--extract folderUid
						SELECT @JSON_Temp = @xml.value('(/root/DocumentLinking/@folderUid)[1]', 'nvarchar(max)')
						IF (@JSON_Temp IS NOT NULL AND @JSON_Temp != '') --#113983
						BEGIN
							SELECT @JSON_Temp = '[' + Concat('"', Replace(@JSON_Temp, ';', '","'), '"') + ']'
							--SELECT @JSON_Temp =  '[' + @JSON_Temp + ']'
						END
						ELSE
							SELECT @JSON_Temp = '[]'
						--add selectedFolderIds property into Tile
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.selectedFolderIds', JSON_QUERY(@JSON_Temp));

						--extract radioButtonReviewAll
						IF (@xml.value('(/root/DocumentLinking/@radioButtonReviewAll)[1]', 'nvarchar(max)') IS NOT NULL)
						BEGIN
							IF (@xml.value('(/root/DocumentLinking/@radioButtonReviewAll)[1]', 'nvarchar(max)') = 'False')
								SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.showAllFolders', CAST(0 as bit));
							ELSE
								SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.showAllFolders', CAST(1 as bit));
						END	
					END
					IF (@TileTypeUid = @ProviderEMTileTypeUid)
					BEGIN		
						--<EandM timespanUid="f2e52f89-7f2c-466a-a878-459f82fa12c9" providersUid="4e274b1b-8b4b-4c19-92bf-8e49cb70d692" value="Bailey, Alex" charttype="bar"/>
						SELECT @JSON_Temp = @xml.value('(/root/EandM/@timespanUid)[1]', 'nvarchar(max)')
						--timespanSelected
						IF (@JSON_Temp IS NOT NULL AND @JSON_Temp != '00000000-0000-0000-0000-000000000000')
							SELECT @JSON_Temp = Concat('[', '"', @JSON_Temp, '"', ']')
						ELSE
							SELECT @JSON_Temp = '[]'
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.timeSpanSelected', JSON_QUERY(@JSON_Temp));

						--selectedProviderIds
						SELECT @JSON_Temp = @xml.value('(/root/EandM/@providersUid)[1]', 'nvarchar(max)')
						IF (@JSON_Temp IS NOT NULL)
						BEGIN
							SELECT @JSON_Temp = Concat('[', '"', Replace(@JSON_Temp, ';', '","'), '"', ']')
							SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.selectedProviderIds', JSON_QUERY(@JSON_Temp));
						END
						ELSE
						BEGIN
							SELECT @JSON_Temp = '[]'
							SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.selectedProviderIds', JSON_QUERY(@JSON_Temp));
						END
						--chartType
						SELECT @JSON_Temp = @xml.value('(/root/EandM/@charttype)[1]', 'nvarchar(max)')
						IF (@JSON_Temp IS NOT NULL)
							SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.chartType', @JSON_Temp);
						ELSE
							SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.chartType', 'pie'); --default to pie even though prm desktop does not save the default of 'pie'
					END
					IF (@TileTypeUid = @UnsentPrescriptonsTileTypeUid)
					BEGIN
						SELECT @JSON_Temp = '[]'
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.providers', JSON_QUERY(@JSON_Temp));
					END
					IF (@TileTypeUid = @VisitNotesTileTypeUid)
					BEGIN		
						SELECT @JSON_Temp = @xml.value('(/root/Filter/@filteruid)[1]', 'nvarchar(max)')
						IF (@JSON_Temp IS NOT NULL AND @JSON_Temp != '00000000-0000-0000-0000-000000000000')
							SELECT @JSON_Temp = Concat('[', '"', @JSON_Temp, '"', ']')
						ELSE
							SELECT @JSON_Temp = '[]'
						--visitNotesFilterSelectedIds
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.visitNotesFilterSelectedIds', JSON_QUERY(@JSON_Temp));
					END
					IF (@TileTypeUid = @WaitingListTileTypeUid)
					BEGIN		
						--<WaitList waitlistuid="8a724e9a-34c7-45d7-8c0c-d05cf79760db" />
						SELECT @JSON_Temp = @xml.value('(/root/WaitList/@waitlistuid)[1]', 'nvarchar(max)')
						IF (@JSON_Temp IS NOT NULL)
							SELECT @JSON_Temp = Concat('[', '"', @JSON_Temp, '"', ']')
						ELSE
							SELECT @JSON_Temp = '[]'
						--selectedWaitingListIds
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.selectedWaitingListIds', JSON_QUERY(@JSON_Temp));
					END
					IF (@TileTypeUid = @ReplicationMonitorTileTypeUid)
					BEGIN
						SELECT @JSON_Temp = @xml.value('(/root)[1]', 'nvarchar(max)')
						IF (@JSON_Temp IS NOT NULL)
							SELECT @JSON_Temp = '[]'
						--selectedCacheMachineIds
						SELECT @JSON_TileDetail = JSON_MODIFY(@JSON_TileDetail, '$.selectedCacheMachineIds', JSON_QUERY(@JSON_Temp));
					END
					--now that a complete TileDetial is built, assign it to the appropriate Tab[].Tiles[
					IF (@ProcessingPrimaryCfg = 1)
						SELECT @JSON_CfgWorking = JSON_MODIFY(@JSON_CfgWorking, 'append $.UserHPConfiguration.Tabs[0].Tiles', JSON_QUERY(@JSON_TileDetail))
					ELSE
						SELECT @JSON_CfgWorking = JSON_MODIFY(@JSON_CfgWorking, 'append $.UserHPConfiguration.Tabs[1].Tiles', JSON_QUERY(@JSON_TileDetail))					
				END --TileTypeUid is not null				

				IF (@ProcessingPrimaryCfg = 1)
					SELECT @ProcessingPrimaryCfg = 0 --finished processing a primary control's cfg so set to work on secondary control presence if any

				SELECT @ProcessLoopCnt = @ProcessLoopCnt + 1
			END --WHILE PROCESS LOOP CNT

			DELETE FROM @UserConfigsToProcessTbl WHERE RelUserPRMDesktopControlUid = @CurrentUserCfgRowUid
		END --WHILE EXISTS FROM @UserConfigsToProcessTbl
	END	

	--store the JSON Blob
	IF (LEN(@JSON_CfgWorking) > 0)
	BEGIN
		INSERT INTO @Stage1_JSONBlob (JSONBlobUid, JSONBlob, JSONUserName, JSONUserPRMUid)	
		VALUES(newid(), @JSON_CfgWorking, @UserConfigName, Convert(VARCHAR(MAX), @CurrentUserUid))
	END
	
	--***DEBUG***
	--SELECT COUNT(*) AS 'STAGE 1 JSON BLOB COUNT' FROM @Stage1_JSONBlob
	--SELECT * FROM @Stage1_JSONBlob order by JSONUserName

	--finished processing current user
	DELETE FROM @ActiveUsers WHERE PersonUid = @CurrentUserUid
END
--******END STAGE 1******

--*****************START STAGE 2*************************
DECLARE @jsonBlob NVARCHAR(MAX)
--These are the end result tables - need to be defined outside of all loops
DECLARE @tileTable TABLE
(
 id uniqueidentifier,
 tileType uniqueidentifier,
 tileData varchar(max)
)

DECLARE @tabTable TABLE
(
 id uniqueidentifier,
 tabName varchar(50), 
 commaDelimitedTileUids varchar(max)
)

DECLARE @homeConfigTable TABLE
(
 id uniqueidentifier,
 userId uniqueidentifier,
 configName varchar(50), 
 commaDelimitedTabUids varchar(max)
)
declare @HdWaitingListUid uniqueidentifier='C68E9C15-86C4-473F-A123-021F6CAE4B97'
declare @HdMessageCountsUid uniqueidentifier='0D30D1A6-D56F-4404-BA37-221531FDFE9B'
declare @HdPromotingInteroperabilityUid uniqueidentifier='8200F46B-A734-409B-9B14-25E8204EED48'
declare @HdUnsentPrescriptionsUid uniqueidentifier='DAD2A405-8AF3-4831-8BC6-52B5C8959558'
declare @HdDocumentLinkingUid uniqueidentifier='973B8745-2931-46B9-8BEE-565385B64E84'
declare @HdVisitNotesUid uniqueidentifier='C7B0C52D-C2B3-43E8-A28F-8031D81F29BA'
declare @HdMessagesUid uniqueidentifier='4EED7F56-6A65-41E5-B568-80A4701C33E1'
declare @HdProviderEMUid uniqueidentifier='1947C8B8-050E-465C-8217-91EF525BC1EC'
declare @HdExternalPatientsUid uniqueidentifier='9AD5225E-4E56-4916-8E27-C7B1DC154520'
declare @HdReplicationMonitorUid uniqueidentifier='DFEE9D62-92D6-4068-845F-CB11C528D95D'
declare @HdAppointmentQueueUid uniqueidentifier='72214A6B-49C5-4D11-8303-D08B23FF58F5'

--populate default tile config variables
declare @HdWaitingListTile nvarchar(max) = '{
    "TileTypeId": "C68E9C15-86C4-473F-A123-021F6CAE4B97",
    "TileTypeName": "Waiting List",
    "TileType": 0,
    "CustomName": "Waiting List",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canMaximize": true,
    "isConfigurable": false,
    "canSort": true,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "canPrint": true,
    "refreshTimer": 5,
    "selectedWaitingListIds": [],
    "includeInactive": false
}'
declare @HdMessageCountsTile nvarchar(max) = '{
    "TileTypeId": "0D30D1A6-D56F-4404-BA37-221531FDFE9B",
    "TileTypeName": "Message Counts",
    "TileType": 0,
    "CustomName": "Message Counts",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "isRequired": false,
    "isUnique": false,
    "isCollapsible": true,
    "isConfigurable": true,
    "canLoad": false,
    "canMaximize": true,
    "canRefresh": true,
    "isLoaded": false,
    "canMoreVert": false,
    "canPrint": true,
    "canSort": true,
    "displayHeader": true,
    "displayFooter": true,
    "refreshTimer": 5,
    "messageFilters": []
}'
declare @HdPromotingInteroperabilityTile nvarchar(max) = '{
    "TileTypeId": "8200F46B-A734-409B-9B14-25E8204EED48",
    "TileTypeName": "Promoting Interoperability",
    "TileType": 0,
    "CustomName": "Promoting Interoperability",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canMaximize": true,
    "isConfigurable": false,
    "canSort": false,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "canPrint": true,
    "selectedProviderIds": [],
    "timeSpanSelected": [
        "f2e52f89-7f2c-466a-a878-459f82fa12c9"
    ],
    "isBillingProvider": true,
    "refreshTimer": 5,
    "customDateRange": {}
}'
declare @HdUnsentPrescriptionsTile nvarchar(max)='{
    "TileTypeId": "DAD2A405-8AF3-4831-8BC6-52B5C8959558",
    "TileTypeName": "Prescription Queue",
    "TileType": 0,
    "CustomName": "Prescription Queue",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canSort": false,
    "canMaximize": true,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "isConfigurable": false,
    "changeRequests": false,
    "refillRequests": false,
    "unsentPrescriptions": false,
    "providers": [],
    "refreshTimer": 5
}'
declare @HdDocumentLinkingTile nvarchar(max)='{
    "TileTypeId": "973B8745-2931-46B9-8BEE-565385B64E84",
    "TileTypeName": "Document Linking",
    "TileType": 0,
    "CustomName": "Document Linking",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canMaximize": true,
    "isConfigurable": false,
    "canSort": false,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "canPrint": true,
    "showAllFolders": true,
    "refreshTimer": 5,
    "selectedFolderIds": []
}'
declare @HdVisitNotesTile nvarchar(max)='{
    "TileTypeId": "C7B0C52D-C2B3-43E8-A28F-8031D81F29BA",
    "TileTypeName": "Visit Notes",
    "TileType": 0,
    "CustomName": "Visit Notes",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canMaximize": true,
    "isConfigurable": false,
    "canSort": false,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "canPrint": true,
    "refreshTimer": 5,
    "visitNotesFilterSelectedIds": []
}'
declare @HdMessagesTile nvarchar(max)='{
    "TileTypeId": "4EED7F56-6A65-41E5-B568-80A4701C33E1",
    "TileTypeName": "Messages",
    "TileType": 0,
    "CustomName": "Messages",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canMaximize": true,
    "isConfigurable": false,
    "canSort": true,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "canPrint": true,
    "filterIds": [],
    "refreshTimer": 5
}'
declare @HdProviderEMTile nvarchar(max)='{
    "TileTypeId": "1947C8B8-050E-465C-8217-91EF525BC1EC",
    "TileTypeName": "Provider E&M",
    "TileType": 0,
    "CustomName": "Provider E&M",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canMaximize": true,
    "isConfigurable": false,
    "canSort": false,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "canPrint": true,
    "selectedProviderIds": [],
    "timeSpanSelected": [
        "f2e52f89-7f2c-466a-a878-459f82fa12c9"
    ],
    "chartType": "bar",
    "refreshTimer": 5,
    "customDateRange": {}
}'
declare @HdExternalPatientsTile nvarchar(max)='{
    "TileTypeId": "9AD5225E-4E56-4916-8E27-C7B1DC154520",
    "TileTypeName": "External Patients",
    "TileType": 0,
    "CustomName": "External Patients",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canMaximize": true,
    "isConfigurable": false,
    "canSort": false,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "canPrint": true,
    "refreshTimer": 5,
    "selectedExternalPatientIds": []
}'
declare @HdReplicationMonitorTile nvarchar(max)='{
    "TileTypeId": "DFEE9D62-92D6-4068-845F-CB11C528D95D",
    "TileTypeName": "Replication Monitor",
    "TileType": 0,
    "CustomName": "Replication Monitor",
    "Description": "",
    "ShortName": "",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canMaximize": true,
    "isConfigurable": false,
    "canSort": false,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "canPrint": true,
    "refreshTimer": 5,
    "selectedCacheMachineIds": []
}'
declare @HdAppointmentQueueTile nvarchar(max)='{
    "TileTypeId": "72214A6B-49C5-4D11-8303-D08B23FF58F5",
    "TileTypeName": "Appointment Queue",
    "TileType": 0,
    "CustomName": "Appointment Queue",
    "Description": "",
    "ShortName": "Appointment Queue",
    "MaxHeight": 700,
    "DefaultState": 0,
    "Width": 2,
    "TileInstanceId": "",
    "canLoad": false,
    "canMoreVert": false,
    "canRefresh": true,
    "canSort": true,
    "canMaximize": true,
    "displayHeader": true,
    "displayFooter": true,
    "isCollapsible": true,
    "isLoaded": false,
    "isRequired": false,
    "isUnique": false,
    "isConfigurable": false,
    "refreshTimer": 5,
    "canPrint": true,
    "calendarFilterIds": [],
    "calendarFilterNames": [],
    "roomFilterIds": [],
    "roomFilterNames": [],
    "appointmentTypeFilterIds": [],
    "appointmentTypesFilterNames": [],
    "appointmentStatusFilterIds": [],
    "appointmentStatusFilterNames": [],
    "displayApptType": true,
    "displayApptReason": true,
    "displayInsurance": true,
    "displayInfoIcons": true,
    "displayCalendar": true,
    "sortOrderings": {
        "SortType": "AppointmentQueueSort",
        "SortOrderingsDetails": [
            {
                "Id": "0AD174FE-014E-496C-8BE1-3ECEDFEBE6D8",
                "Name": "Appointment Time",
                "ParameterValue": "StartDateTime",
                "IsAscending": true
            }
        ]
    }
}'

WHILE(EXISTS(SELECT * FROM @Stage1_JSONBlob))
BEGIN
	SELECT TOP 1 @CurrentDataRowUid = JSONBlobUid, @jsonBlob = JSONBlob FROM @Stage1_JSONBlob

	--Make sure this is inside homepage person loop
	declare @HomepageId uniqueidentifier = newId()
	declare @PersonUid uniqueidentifier
	declare @HomepageName varchar(50)
	declare @commaDelimitedTabIdsToInsert varchar(max) = ''
	
	declare @UidString varchar(36)
	select @UidString = value from OPENJSON(@jsonBlob, '$.UserHPConfiguration')
	where [key]='UserPRMUid'
	select @PersonUid=CAST(@UidString AS UNIQUEIDENTIFIER)

	select @HomepageName=value from OPENJSON(@jsonBlob, '$.UserHPConfiguration')
	where [key]='HomepageName'

	--begin tab loop
	declare @Tabs Table(
		id uniqueidentifier,
		DisplayOrder int,
		Name varchar(50),
		comment varchar(max),
		HasTiles bit,
		Tiles nvarchar(max)
	)
	insert into @Tabs (id, DisplayOrder, Name, comment, HasTiles, Tiles)
	select newid(), row_number() over (order by [key]), Name, comment, HasTiles, Tiles from OPENJSON(@jsonBlob, '$.UserHPConfiguration.Tabs')
	with (
		[key] int,
		Name varchar(50) '$.Name',
		comment varchar(max) '$.comment',
		HasTiles bit '$.HasTiles',
		Tiles nvarchar(max) '$.Tiles' as json
	)

	--Tab Data to insert
	declare @currentTabId uniqueidentifier
	declare @currentTabName varchar(50)
	declare @currentTabComment varchar(max)
	declare @currentTileList nvarchar(max)

	WHILE ((select count(*) from @Tabs) > 0)
	BEGIN
		SELECT TOP 1 
		@currentTabId=id,
		@currentTabName=Name,
		@currentTabComment=comment,
		@currentTileList=Tiles
		from @Tabs
		order by DisplayOrder

		declare @commaDelimitedTileIdsToInsert varchar(max) = ''

		--Tile Data to insert
		declare @currentTileId uniqueidentifier
		declare @currentTileTypeId uniqueidentifier
		declare @importedTileConfig nvarchar(max)

		--begin tile loop
		declare @Tiles Table(
			id uniqueidentifier,
			DisplayOrder int,
			TileTypeUid uniqueidentifier,
			TileData nvarchar(max)
		)
		insert into @Tiles (id, DisplayOrder, TileTypeUid, TileData)
		select newid(), row_number() over (order by [key]), TileTypeUid, TileData from OPENJSON(@currentTileList)
		with (
			[key] int,
			TileTypeUid uniqueidentifier '$.TileTypeUid',
			TileData nvarchar(max) '$' as json
		)

		WHILE ((select count(*) from @Tiles) > 0)
		BEGIN
			SELECT TOP 1 
			@currentTileId=id,
			@currentTileTypeId=TileTypeUid,
			@importedTileConfig=TileData
			from @Tiles
			order by DisplayOrder

			declare @tileConfig nvarchar(max) =
			CASE
				WHEN @currentTileTypeId=@HdWaitingListUid THEN @HdWaitingListTile
				WHEN @currentTileTypeId=@HdMessageCountsUid THEN @HdMessageCountsTile
				WHEN @currentTileTypeId=@HdPromotingInteroperabilityUid THEN @HdPromotingInteroperabilityTile
				WHEN @currentTileTypeId=@HdUnsentPrescriptionsUid THEN @HdUnsentPrescriptionsTile
				WHEN @currentTileTypeId=@HdDocumentLinkingUid THEN @HdDocumentLinkingTile
				WHEN @currentTileTypeId=@HdVisitNotesUid THEN @HdVisitNotesTile
				WHEN @currentTileTypeId=@HdMessagesUid THEN @HdMessagesTile
				WHEN @currentTileTypeId=@HdProviderEMUid THEN @HdProviderEMTile
				WHEN @currentTileTypeId=@HdExternalPatientsUid THEN @HdExternalPatientsTile
				WHEN @currentTileTypeId=@HdReplicationMonitorUid THEN @HdReplicationMonitorTile
				WHEN @currentTileTypeId=@HdAppointmentQueueUid THEN @HdAppointmentQueueTile
			END

			set @tileConfig = JSON_MODIFY(@tileConfig,'$.TileInstanceId', CAST(@currentTileId as varchar(100)))

			--perform other modifys here - tile specific adjustments
			set @tileConfig =
			CASE
				WHEN @currentTileTypeId=@HdWaitingListUid THEN
					JSON_MODIFY(@tileConfig,'$.selectedWaitingListIds', JSON_Query(@importedTileConfig, '$.selectedWaitingListIds'))
				WHEN @currentTileTypeId=@HdMessageCountsUid THEN
					JSON_MODIFY(@tileConfig,'$.messageFilters', JSON_Query(@importedTileConfig, '$.messageFilters'))
				WHEN @currentTileTypeId=@HdPromotingInteroperabilityUid THEN
					--#114271
					JSON_MODIFY(
						JSON_MODIFY(@tileConfig,'$.selectedProviderIds', JSON_Query(@importedTileConfig, '$.selectedProviderIds')),
						'$.timeSpanSelected', JSON_QUERY(@importedTileConfig, '$.timeSpanSelected')
					)
				WHEN @currentTileTypeId=@HdUnsentPrescriptionsUid THEN
					JSON_MODIFY(@tileConfig,'$.selectedCacheMachineIds', JSON_Query(@importedTileConfig, '$.selectedCacheMachineIds'))
				WHEN @currentTileTypeId=@HdDocumentLinkingUid THEN  --#114277
					JSON_MODIFY( 
						JSON_MODIFY(@tileConfig,'$.selectedFolderIds', JSON_Query(@importedTileConfig, '$.selectedFolderIds')),
							'$.showAllFolders', 
							CASE JSON_VALUE(@importedTileConfig, '$.showAllFolders') 
							WHEN 'true' THEN 
								CAST(1 as bit) ELSE CAST(0 as bit)
							END
					)
				WHEN @currentTileTypeId=@HdVisitNotesUid THEN
					JSON_MODIFY(@tileConfig,'$.visitNotesFilterSelectedIds', JSON_Query(@importedTileConfig, '$.visitNotesFilterSelectedIds'))
				WHEN @currentTileTypeId=@HdMessagesUid THEN
					JSON_MODIFY(@tileConfig,'$.filterIds', JSON_Query(@importedTileConfig, '$.filterIds'))
				WHEN @currentTileTypeId=@HdProviderEMUid THEN
					JSON_MODIFY(
						JSON_MODIFY(
							JSON_MODIFY(@tileConfig,'$.selectedProviderIds', JSON_Query(@importedTileConfig, '$.selectedProviderIds')),
							'$.timeSpanSelected',
							JSON_QUERY(@importedTileConfig, '$.timeSpanSelected')
						),
						'$.chartType',
						JSON_VALUE(@importedTileConfig, '$.chartType')
					)
				WHEN @currentTileTypeId=@HdExternalPatientsUid THEN 
					JSON_MODIFY(@tileConfig,'$.selectedExternalPatientIds', JSON_Query(@importedTileConfig, '$.selectedExternalPatientIds'))
				WHEN @currentTileTypeId=@HdReplicationMonitorUid THEN
					JSON_MODIFY(@tileConfig,'$.selectedCacheMachineIds', JSON_Query(@importedTileConfig, '$.selectedCacheMachineIds'))
				WHEN @currentTileTypeId=@HdAppointmentQueueUid THEN
					JSON_MODIFY(
						JSON_MODIFY(
							JSON_MODIFY(
								JSON_MODIFY(@tileConfig,'$.calendarFilterIds', JSON_Query(@importedTileConfig, '$.calendarFilterIds')),
								'$.appointmentStatusFilterIds',
								JSON_QUERY(@importedTileConfig, '$.appointmentStatusFilterIds')
							),
							'$.roomFilterIds',
								JSON_QUERY(@importedTileConfig, '$.roomFilterIds')
						),
						'$.appointmentTypeFilterIds',
						JSON_QUERY(@importedTileConfig, '$.appointmentTypeFilterIds')
					)
				END
			--build up tile list comma delimited string
			if (@commaDelimitedTileIdsToInsert is not null) AND (LEN(@commaDelimitedTileIdsToInsert) > 0)
			Begin
				set @commaDelimitedTileIdsToInsert = Concat(@commaDelimitedTileIdsToInsert, ',' , @currentTileId)
			End
			else
			BEGIN
				set @commaDelimitedTileIdsToInsert = @currentTileId
			END

			--insert into end tile table
			insert into @tileTable(id, tileType, tileData)
			values(@currentTileId, @currentTileTypeId, @tileConfig)

			delete @Tiles where id=@currentTileId
		END -- end tile loop

		if (@commaDelimitedTabIdsToInsert is not null) AND (LEN(@commaDelimitedTabIdsToInsert) > 0)
		Begin
			set @commaDelimitedTabIdsToInsert = Concat(@commaDelimitedTabIdsToInsert, ',' , @currentTabId)
		End
		else
		BEGIN
			set @commaDelimitedTabIdsToInsert = @currentTabId
		END

		--insert into end tab table
		insert into @tabTable(id, tabName, commaDelimitedTileUids)
		values(@currentTabId, @currentTabName, @commaDelimitedTileIdsToInsert)

		delete @Tabs where id=@currentTabId
	END -- end tab loop

	--Make sure this is inside homepage person loop
	insert into @homeConfigTable(id, userId, configName, commaDelimitedTabUids)
	values(@HomepageId, @PersonUid, @HomepageName, @commaDelimitedTabIdsToInsert)

	DELETE FROM @Stage1_JSONBlob WHERE JSONBlobUid = @CurrentDataRowUid
END --END WHILE PROCESSING JSONBLOBS

-- remove configs with no tabs
delete @homeConfigTable where commaDelimitedTabUids=''
--*****************END STAGE 2 *******************************

--*****************BEGIN STAGE 3 *****************************
declare @homepageConfigId uniqueidentifier
declare @userId uniqueidentifier
declare @commaDelimitedTabUids varchar(max) -- loop from this
declare @HomepageNameFinal varchar(50)

--loop based on values in @homeConfigTable
WHILE ((select count(*) from @homeConfigTable) > 0)
BEGIN
	SELECT TOP 1 
	@homepageConfigId=id,
	@HomepageNameFinal=configName,
	@userId=userId,
	@commaDelimitedTabUids=commaDelimitedTabUids
	from @homeConfigTable

	--GroupUiConfig insert logic
	insert into ListGroupUIConfiguration(GroupUIConfigurationUid, DataTypeId, Name, Notes, Inactive, UserDefined, LastModifiedDate)
	values(
		@homepageConfigId,
		1,
		@HomepageNameFinal,
		'Homepage user migration config',
		0,
		1,
		GETUTCDATE()
	)

	--relate to user
	insert into RelUserPRMGroupUIConfiguration(RelUserPRMGroupUIConfigurationUid, UserUid, GroupUIConfigurationUid, LastModifiedDate)
	values(
		NEWID(),
		@userId,
		@homepageConfigId,
		GETUTCDATE()
	)

	DECLARE @tabIdList TABLE
	(
	 id uniqueidentifier
	)

	if @commaDelimitedTabUids <> ''
	BEGIN
		;WITH CTE
		 AS (
		 SELECT Split.a.value('.', 'NVARCHAR(MAX)') id
		 FROM
		 (
			 SELECT CAST('<X>'+REPLACE(@commaDelimitedTabUids, ',', '</X><X>')+'</X>' AS XML) AS String
		 ) AS A
		 CROSS APPLY String.nodes('/X') AS Split(a))
		 INSERT INTO @tabIdList
				SELECT A.id
				FROM CTE A
	END
	
	--tab logic setup
	declare @tabId uniqueidentifier
	declare @tabName varchar(50)
	declare @commaDelimitedTileUids varchar(max)
	declare @currentTabIndex int = 0

	WHILE ((select count(*) from @tabIdList) > 0) -- tab loop
	BEGIN
		select top 1 @tabId=id from @tabIdList

		SELECT TOP 1 
		@tabName=tabName,
		@commaDelimitedTileUids=commaDelimitedTileUids
		from @tabTable
		where id = @tabId

		--tab insert logic
		insert into ListUIConfiguration(UIConfigurationUid, DataTypeId, Name, Notes, Data, Inactive, UserDefined, LastModifiedDate)
		values (
			@tabId,
			2,
			@tabName,
			'Homepage user migration tab',
			'{}',
			0,
			1,
			GETUTCDATE()
		)

		DECLARE @tileIdList TABLE
		(
		 id uniqueidentifier
		)

		if @commaDelimitedTileUids <> ''
		BEGIN
			;WITH CTE
			 AS (
			 SELECT Split.a.value('.', 'NVARCHAR(MAX)') id
			 FROM
			 (
				 SELECT CAST('<X>'+REPLACE(@commaDelimitedTileUids, ',', '</X><X>')+'</X>' AS XML) AS String
			 ) AS A
			 CROSS APPLY String.nodes('/X') AS Split(a))
			 INSERT INTO @tileIdList
					SELECT A.id
					FROM CTE A
		END

		--tile logic setup
		declare @tileId uniqueidentifier
		declare @tileType uniqueidentifier
		declare @tileData varchar(max)
		declare @currentTileIndex int = 0

		WHILE ((select count(*) from @tileIdList) > 0) -- tile loop
		BEGIN
			select top 1 @tileId=id from @tileIdList

			SELECT
			@tileType=tileType,
			@tileData=tileData
			from @tileTable
			where id = @tileId

			--tile insert logic
			insert into ListTileConfiguration(TileConfigurationUid, UIConfigurationUid, TileTypeUid, Data, DisplayOrder, Inactive, UserDefined, LastModifiedDate)
			values (
				@tileId,
				@tabId,
				@tileType,
				@tileData,
				@currentTileIndex,
				0,
				1,
				GETUTCDATE()
			)
			set @currentTileIndex = @currentTileIndex + 1
			delete @tileIdList where id = @tileId
		END --tile loop

		insert into RelUIConfigurationGroupUIConfiguration(
			RelUIConfigurationGroupUIConfigurationUid, 
			GroupUIConfigurationUid, 
			UIConfigurationUid, 
			DisplayOrder,
			UserDefined,
			LastModifiedDate
		)
		values(
			newid(),
			@homepageConfigId,
			@tabId,
			@currentTabIndex,
			1,
			GETUTCDATE()
		)
		set @currentTabIndex = @currentTabIndex + 1
		delete @tabIdList where id = @tabId
	END --tab loop

	delete @homeConfigTable where id = @homepageConfigId
END
--*****************END STAGE 3 *******************************